/* Copyright (C) 2011-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef __VNCDISCOVERER_H__
#define __VNCDISCOVERER_H__

#include <vncdiscoverysdktypes.h>
#include <vncdiscoverersupportapi.h>
#include <stddef.h>

/**
* \file vncdiscoverer.h
*
* This is the main header file for the VNC Discoverer Interface.  
* To implement the interface you need only include this file.
*
* This file defines the VNC Discoverer Interface, the data structure associated
* with it and the sole entry point for the Discoverer DLL or shared object.
*
* \see VNCDiscovererInitializeType, VNCDiscovererInterface
* \see vncdiscoverersupportapi.h, vncdiscoverysdktypes.h
*/

/**
 * \mainpage VNC Discoverer Native Interface
 *
 * \section section_overview Overview
 *
 * The Discovery SDK is for use within viewer applications and allows
 * discovering of VNC Servers. The Discovery SDK will detect devices that
 * contain a VNC Server and provide a vnccmd string which can be passed to the VNC
 * Viewer SDK to establish a VNC connection to the device. The VNC Discovery SDK
 * can run in parallel with the VNC Viewer SDK and discovering of new servers
 * can happen while the Viewer is connected to an existing server. Several VNC
 * connections can be run in parallel by using several instances of the VNC
 * Viewer SDK. Please consult the VNC Viewer SDK for more information.
 *
 * The Discovery SDK abstracts away from the medium to which the connection to the server 
 * is made. It allows plug-ins to be loaded dynamically. A plugin is called a 
 * Discoverer and the user of the SDK, the client application, can load a specific plug-in by
 * specifying its type.
 *
 * Pluggable Discoverers provide a means of extending the Discovery SDK to
 * support different types of Discoverers. This allows the applications to
 * discover VNC Servers through different methods, using different
 * transports and discovering on as many interfaces as is required.
 *
 * This document covers the interface between the VNC Discovery SDK and the
 * plug-in Discoverers. If you want to learn about the Discovery SDK client API
 * please see the "VNC Discovery SDK - Native Interface" document.
 *
 * The Discoverer implementation must implement means of finding VNC Servers,
 * referred to as 'entities' in the Discovery SDK context. They also must be
 * able to retrieve information about these entities, like the unique
 * identifier, name, type and the VNC command. Discoverers can implement other
 * part of the interface to allow extra operations on the entities, as it is
 * most suited to the particular case.
 *
 * If you are interested in separating the Discoverer logic from the device
 * detection logic, you might want to write, or use, a Device Provider. More
 * information about Device Providers can be found in "VNC Discovery SDK -
 * Native Device Provider Plug-in Interface".
 * 
 * To implement a Discoverer you must create a DLL shared object that exports
 * an entry point named VNCDiscovererInitialize. The entry point must conform
 * to the specification of VNCDiscovererInitializeType as described below.
 * The Discoverer must also provide an implementation for some of the methods
 * in the VNCDiscovererInterface structure. As a minimum it must implement
 * VNCDiscovererSetContext, VNCDiscovererStart, VNCDiscovererStop,
 * VNCDiscovererFetchEntityValue, VNCDiscovererDestroyEntity and
 * VNCDiscovererDestroy.
 *
 * The name of the Discoverer DLL or shared object must be
 * vncdiscoverer-somename.dll in Windows, or libvncdiscoverer-somename.so in
 * Linux. In that case the type of Discoverer is 'somename' and this will be
 * used by the client application to ask the SDK to load it.
 *
 * The Discoverer communicates back to the application through the
 * VNCDiscovererSupportAPI.
 *
 * The SDK can monitor for the Discoverer event handles (file descriptors in
 * Unix) for any activity. When some activity occurs, the discoverer will be
 * notified and asked to handle the event. Initially only one handle per
 * discoverer has been allowed, and this handle was retrieved by the SDK
 * through VNCDiscovererGetEventHandle. When an event occurs,
 * VNCDiscovererHandleEvent would be called. Later on support for multiple
 * event handles has been added. The event handles are retrieved through
 * VNCDiscovererGetMultipleEventHandlesEx, and events are notified through
 * VNCDiscovererHandleSingleEvent. A discoverer must define at most one of
 * VNCDiscovererGetEventHandle or VNCDiscovererGetMultipleEventHandlesEx. For
 * each of the the corresponding handling mechanism must be defined as well. If
 * both methods of retrieval are defined, then the discoverer will not be
 * loaded successfully.
 *
 * The discoverer can also request to be notified of a timeout through
 * VNCDiscovererHandleTimeout. The timer is set through VNCDiscovererSetTimer.
 * If multiple handles are used, then a timeout can be given for each handle.
 * VNCDiscovererSetTimerForHandle must be used in that case, and the timeout is
 * received in VNCDiscovererHandleTimeoutForEventHandle. The Discoverer must
 * implement only one of the timeout handlers, otherwise loading will fail.
 *
 * If your platform supports C++ and the STL, then you may wish to use the
 * VNCDiscovererImpl class as your starting point, rather than implementing
 * this API directly. These classes may be found in the common directory of the
 * Discoverer source tree.
 *
 * \section section_legal Legal information
 *
 * Copyright &copy; 2002-2018 RealVNC Ltd.  All Rights Reserved.
 *
 * RealVNC and VNC are trademarks of RealVNC Limited and are protected by
 * trademark registrations and/or pending trademark applications in the
 * European Union, United States of America and other jurisdictions.
 *
 * Protected by UK patents 2481870, 2491657; US patents 8760366, 9137657; EU patent 2652951.
 *
 * \see VNCDiscovererInitializeType, VNCDiscovererInterface, VNCDiscovererSupportAPI
 */

#ifdef __cplusplus
extern "C"
{
#endif


/**
 * \brief Called by the SDK to set a context that will be presented in the
 * supportAPI.
 *
 * The implementation should associate the requested Discoverer with the
 * given context. This context should be used henceforth in the support API.
 *
 * \param pDiscoverer The Discoverer that will use the new context
 * \param pContext The new context
 *
 * \see VNCDiscovererSupportAPI
 */
typedef void VNCCALL
VNCDiscovererSetContext(VNCDiscoverer *pDiscoverer, void *pContext);

/**
 * \brief Called by the SDK to get the event handle of the Discoverer.
 *
 * The handle will be monitored by the SDK and whenever an event happens
 * VNCDiscovererHandleEvent will be called. This method will be called each
 * time the Discovery SDK will listen for new event. The handle value should
 * not change between calls to this function.
 *
 * The event handle will not be monitored when the Discoverer is not started. 
 *
 * If a Discoverer needs to pass multiple handles, then this method must not
 * be implemented, and VNCDiscovererGetMultipleEventHandlesEx should be used
 * instead. Having both this method and VNCDiscovererGetMultipleEventHandlesEx
 * implemented will result in the discoverer not being loaded.
 *
 * \param pDiscoverer The Discoverer for which to monitor the handle.
 *
 * \return Pointer to the event handle of the Discoverer. If the Discoverer
 *    chooses not to provide a handle, then this should be NULL. If the Discoverer
 *    doesn't wait for any event at the moment (but might wait later on), then the
 *    value can be VNCDiscovererNoEventHandle.
 *
 * \see VNCDiscovererHandleEvent, VNCDiscovererSetTimer,
 * VNCDiscovererHandleTimeout
 * \see VNCDiscovererGetMultipleEventHandlesEx
 */
typedef VNCDiscovererEventHandle *VNCCALL
VNCDiscovererGetEventHandle(VNCDiscoverer *pDiscoverer);

/**
 * \brief Method called by the SDK whenever and event happens on the handle
 * provided by the Discoverer.
 *
 * This method will be called to notify the Discoverer that an event happened
 * on the handler provided. If the Discoverer is not providing a handle to the
 * SDK then it doesn't need to implement this method. Otherwise an
 * implementation is necessary.
 *
 * During this call it is safe to change the event handle, or to reset the
 * timer for the Discoverer.
 *
 * This method will not be called if using multiple handles. The Discoverer
 * must implement VNCDiscovererHandleSingleEvent instead.
 *
 * \note If you don't stop the timer, or reset it, you will be notified when
 *      the original timeout elapses.
 *
 * \param pDiscoverer The Discoverer for which the event happened.
 *
 * \see VNCDiscovererInitializeType
 * \see VNCDiscovererGetEventHandle
 * \see VNCDiscovererGetMultipleEventHandlesEx, VNCDiscovererHandleSingleEvent
 */
typedef void VNCCALL
VNCDiscovererHandleEvent(VNCDiscoverer *pDiscoverer);

/**
 * \brief Method called by the SDK whenever the timeout requested by the
 * Discoverer is reached. 
 *
 * This method will be called to notify the Discoverer that the timeout
 * requested was reached. 
 *
 * If the Discoverer is not asking for timer access from the
 * SDK then it doesn't need to implement this method. Otherwise an
 * implementation is necessary.
 *
 * During this call it is safe to change the event handle for the Discoverer.
 *
 * If the Discoverer wishes to have further timer callbacks it must set the
 * timeout request again.
 *
 * This method will not be called if using multiple handles. The Discoverer
 * must implement VNCDiscovererHandleTimeoutForEventHandle instead.
 *
 * \param pDiscoverer The Discoverer for which the target time was reached.
 *
 * \see VNCDiscovererSetTimer
 * \see VNCDiscovererGetEventHandle
 * \see VNCDiscovererGetMultipleEventHandlesEx,
 * VNCDiscovererHandleTimeoutForEventHandle
 */
typedef void VNCCALL
VNCDiscovererHandleTimeout(VNCDiscoverer *pDiscoverer);

/**
 * \brief Called by the SDK to get the event handle of an entity.
 *
 * The handle will be monitored by the SDK and whenever an event happens
 * VNCDiscovererHandleEntityEvent will be called. This method will be called
 * each time the Discovery SDK will listen for new event. The handle value
 * should not change between calls to this function.
 *
 * The event handle will not be monitored when the Discoverer is not started. 
 *
 * Although the Discoverer can have multiple event handles for each Discoverer
 * instance, the Entity can only have one event handle per instance. If an
 * Entity would want to monitor more than one event handle, then the
 * recommended way of doing this is through the Discoverer instance. The
 * Discoverer would filter any calls to VNCDiscovererHandleSingleEvent.
 *
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The Entity that might be waiting for an event.
 *
 * \return Pointer to the event handle of the entity. If the entity
 *    chooses not to provide a handle, then this should be NULL. If the entity
 *    doesn't wait for any event at the moment (but might wait later on), then the
 *    value can be VNCDiscovererNoEventHandle.
 */
typedef VNCDiscovererEventHandle *VNCCALL
VNCDiscovererGetEntityEventHandle(VNCDiscoverer *pDiscoverer,
                          VNCDiscovererEntity *pEntity);

/**
 * \brief Method called by the SDK whenever and event happens on a handle
 * provided by one of the Entities of the Discoverer.
 *
 * This method will be called to notify an Entity of the Discoverer that an 
 * event happened on the handler provided. If the Entities of this Discoverer 
 * are not providing a handle to the SDK then the implementation doesn't need 
 * provide this method. Otherwise an implementation is necessary.
 *
 * During this call it is safe to change the event handle, or reset the timer
 * for the Entity.
 *
 * \note If you don't stop the timer, or reset it, you will be notified when
 *      the original timeout elapses.
 *
 * \param pDiscoverer The Discoverer that owns the Entity for which the event 
 *                    happened.
 * \param pEntity The Entity for which the event happened.
 *
 * \see VNCDiscovererNotifyEntityAppeared
 */
typedef void VNCCALL
VNCDiscovererHandleEntityEvent(VNCDiscoverer *pDiscoverer,
                            VNCDiscovererEntity *pEntity);

/**
 * \brief Method called by the SDK whenever the timeout requested by an Entity 
 * of the Discoverer is reached.
 *
 * This method will be called to notify an Entity that the timeout it 
 * requested was reached. 
 *
 * If the Entities of the Discoverer are not asking for timer access from the 
 * SDK then it doesn't need to implement this method. Otherwise an
 * implementation is necessary.
 *
 * During this call it is safe to change the event handle for the Entity.
 *
 * Before the SDK calls this method it calls VNCDiscovererSetNoTargetTime on
 * the target time. So if the Entity doesn't want any more callbacks it
 * doesn't need to do anything. If it wants to be called back again, it should
 * set the new target time.
 *
 * \param pDiscoverer The Discoverer for which the target time was reached.
 *
 * \see VNCDiscovererSetEntityTimer
 */
typedef void VNCCALL
VNCDiscovererHandleEntityTimeout(VNCDiscoverer *pDiscoverer,
                            VNCDiscovererEntity *pEntity);

/**
 * \brief Sets a specific property on a Discoverer instance (synchronous).
 *
 * If the Discoverer supports different modes of operations, or it has
 * different properties that can be set by the Application, then it should
 * implement this method to allow the Application to set those properties.
 *
 * The Discoverer should be able to discover without any settings being
 * necessary. So it should have a default mode of operation. This is to allow
 * simple client applications that don't need to know what settings are
 * necessary to start discovering.
 *
 * \param pDiscoverer The Discoverer for which to set the property.
 * \param pProperty The string for the property to set (the key). Ownership of
 *                  the string stays with the SDK.
 * \param pValue The string for the value to set to (or NULL if to unset).
 *               Ownership of the string stays with the SDK.
 *
 * \return VNCDiscoverySDKErrorNone if the property was successfully set,
 *         VNCDiscoverySDKErrorNotSupported if the Discoverer doesn't support
 *         this property, or another error specific for the Discoverer.
 *
 * \see VNCDiscoverySDKError
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscovererSetProperty(VNCDiscoverer *pDiscoverer,
                          const char *pProperty,
                          const char *pValue);

/**
 * \brief Gets a specific property from a Discoverer instance (synchronous).
 *
 * If the Discoverer wants to be able to report to the Application different
 * settings/modes of operations etc., then it has to implement this method to do
 * so.
 *
 * \param pDiscoverer The Discoverer for which the property is to be retrieved.
 * \param pPropery The string representing the property to be retrieved. The
 *                 ownership of the string stays with the SDK.
 * \param pValue The pointer that needs to be set to the result string. The
 *               ownership of the string is passed to the SDK, therefore the
 *               Discoverer should use VNCDiscovererAllocString from the
 *               Support API to allocate the memory for the string. If the
 *               Property is not set the pointer returned is NULL (and the
 *               error is VNCDiscoverySDKErrorNone).
 *
 * \return VNCDiscoverySDKErrorNone if the property was successfully retrieved,
 *         VNCDiscoverySDKErrorNotSupported if the Discoverer doesn't support
 *         this property, or another error specific for the Discoverer.
 *
 * \see VNCDiscoverySDKError, VNCDiscovererSupportAPI
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscovererGetProperty(VNCDiscoverer *pDiscoverer,
                          const char *pProperty,
                          char **pValue);

/**
 * \brief Retrieves synchronously an array of extra Viewer parameters used for 
 * a particular Discoverer.
 *
 * In addition to the VNC command, the VNC Viewer sometimes needs to set a number
 * of parameters to connect to a VNC Server. They are needed for some types of
 * bearers. This method retrieves an array of all such Viewer Parameters, as 
 * key-value pairs, needed for establishing the connection. The ownership of
 * the array is passed to the SDK.
 *
 * \param pEntity The Entity for which to retrieve the Viewer parameters.
 * \param pParametersArray A pointer that will be set to the place where the
 *        result array will be stored. Ownership of the array is passed to the
 *        SDK, therefore the array should be allocated with
 *        VNCDiscovererAlloKeyValPairArray from the Support API.
 * \param pParametersArraySize A pointer to where the size of the array will be
 *        stored.
 *
 * \return An error code that specifies if the parameters were retrieved 
 * successfully, or some error occurred.
 *
 * \see VNCDiscoverySDKError, VNCDiscovererSupportAPI
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscovererGetViewerParameters(VNCDiscoverer *pDiscoverer,
                            VNCDiscovererEntity *pEntity, 
                            VNCDiscoverySDKKeyValPair **pParametersArray,
                            size_t *pParametersArraySize); 

/**
 * \brief Starts the discovery process.
 *
 * Starts the discovery process for a specific Discoverer instance. The
 * implementation should report the outcome of the start call through
 * VNCDiscovererNotifyStarted from the Support API. This is an asynchronous
 * call, so the Discoverer implementation should return from this method as
 * quickly as possible and it can initiate the notification at a later point.
 *
 * Any operations on any entities are expected to fail when the Discoverer is
 * stopped.
 *
 * If the Discoverer provided and event handler, once this method returns the
 * Discoverer shouldn't change the Event Handle. At this point monitoring of
 * the Event Handle starts, unless it is set to an invalid value (for Linux a
 * valid value is between 0 and FD_SETSIZE).
 *
 * Also if there are Event Handles of Entities present from a previous run of
 * the Discoverer, the SDK will re-start monitoring those Handles.
 *
 * The Discoverer might be stopped if the Event Handle is valid and the SDK is
 * unable to monitor it (e.g. because of an out of memory error).
 * 
 * \param pDiscoverer The Discoverer instance that is supposed to start.
 *
 * \see VNCDiscovererSupportAPI
 */
typedef void VNCCALL
VNCDiscovererStart(VNCDiscoverer *pDiscoverer);

/**
 * \brief Stops the discovery process.
 *
 * Stops the discovery process for a specific Discoverer instance. The
 * implementation should confirm stopping by calling
 * VNCDiscovererNotifyStopped, from the Support API. 
 *
 * Any operations on any entities are expected to fail when the Discoverer is
 * stopped. This is an asynchronous call, so the Discoverer implementation 
 * should return from this method as quickly as possible and it can initiate 
 * the notification at a later point.
 *
 * A Discoverer is expected to be able to re-Start at a later point.
 * 
 * \param pDiscoverer The Discoverer instance that needs to stop.
 *
 * \see VNCDiscovererSupportAPI
 */
typedef void VNCCALL
VNCDiscovererStop(VNCDiscoverer *pDiscoverer);

/**
 * \brief Fetches asynchronously a specific entity value.
 *
 * This method asks the Discoverer to retrieve asynchronously a value for a
 * specific key of an Entity. The Discoverer implementation will return the
 * results of the retrieve through VNCDiscovererReturnEntityValue from the
 * Support API. 
 *
 * The Discoverer should return from this method very quickly.
 * 
 * The Discovery SDK requires the Discoverer implementation to support the keys
 * "id", "type", "name" and "vnccmd". Out of these the Discovery SDK requires
 * that the values for the keys "id" and "type" can be retrieved straight away,
 * so they need to be readily available as long as the Entity exists. Other 
 * values can take longer to be retrieved.
 *
 * This call has a unique request ID. The Discoverer needs to use this ID
 * when calling VNCDiscovererReturnEntityValue. If a Discoverer doesn't support
 * this feature it should pass VNCDiscoverySDKNoRequestId to the Support API.
 *
 * \note The existence of an entity does not need to be definite, even if  
 * the Discoverer successfully provides a vnccmd. It does not need to guarantee 
 * that a VNC Server is present and listening for incoming connections. False 
 * positives may be reported by some Discoverers. It is ideal, but not
 * necessary, for Discoverers to ensure the VNC Server is ready to receive
 * incoming connections.
 *
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The Entity from which to retrieve the value.
 * \param pKey The key for which to retrieve the value. This ownership of the 
 *             string is passed to the Discoverer.
 * \param requestId The request ID that should be passed when reporting the
 *                  result.
 *
 * \see VNCDiscovererSupportAPI
 */
typedef void VNCCALL
VNCDiscovererFetchEntityValue(VNCDiscoverer *pDiscoverer,
                              VNCDiscovererEntity *pEntity,
                              char *pKey,
                              VNCDiscoverySDKRequestId requestId);

/**
 * \brief Gets some binary data owned by an entity.
 *
 * This method asks the Discoverer to get a pointer to some binary data owned
 * by the entity. To identify the data a key is used. This key might have been
 * passed as a result of a Fetch Entity Value, or be a documented key.
 *
 * This is not a mandatory method and Discoverers don't need to implement it if
 * they don't have any binary data that can be useful to the client
 * application.
 *
 * If the binary data can't be found then the return size will be 0 and the
 * pointer will point to NULL.
 *
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The Entity from which to retrieve the binary data.
 * \param pKey The key of the data. This key is either something documented by
 *          the Discoverer, or retrieved from the entity as a result of a Fetch
 *          Entity Value.
 * \param pData This will point to the binary data on return, or NULL if the
 *          data can't be found.
 *
 * \return The size of the binary data. If the data can't be found the size is
 *          0.
 */
typedef size_t VNCCALL
VNCDiscovererGetEntityBinaryData(VNCDiscoverer *pDiscoverer,
                              VNCDiscovererEntity *pEntity,
                              const char *pKey,
                              const vnc_uint8_t **pData);

/**
 * \brief Post asynchronously a specific entity value.
 *
 * This method asks the Discoverer to post asynchronously a value for a
 * specific key of an Entity. The Discoverer implementation will notify of the 
 * results of the post through VNCDiscovererNotifyPostEntityValue from the
 * Support API. 
 *
 * The Discoverer should return from this method very quickly.
 * 
 * This call has a unique request ID. The Discoverer needs to use this ID
 * when calling VNCDiscovererNotifyPostEntityValue. If a Discoverer doesn't 
 * support this feature it should pass VNCDiscoverySDKNoRequestId to the i
 * Support API.
 *
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The Entity for which to set the value.
 * \param pKey The key for which to retrieve the value. The ownership of the
 *             string is passed to the Discoverer.
 * \param pValue The value to which to set the key. A NULL pointer means
 *               un-setting the value. The ownership of the string is passed to
 *               the Discoverer.
 * \param requestId The request ID that should be passed when reporting the
 *                  result.
 *
 * \see VNCDiscovererSupportAPI
 */
typedef void VNCCALL
VNCDiscovererPostEntityValue(VNCDiscoverer *pDiscoverer,
                            VNCDiscovererEntity *pEntity,
                            char *pKey,
                            char *pValue,
                            VNCDiscoverySDKRequestId requestId);

/**
 * \brief Cancels and outstanding fetch/post request.
 *
 * Method used by the Discovery SDK to ask the Discoverer to cancel an
 * outstanding request. The Discoverer implementation may omit this method, in
 * which case it means that the Discoverer does not support cancellation.
 *
 * This method is used for example when a time-out occurs.
 *
 * \param pDiscoverer The Discoverer instance that has the outstanding request.
 * \param requestId The ID of the fetch/post request.
 *
 * \return The outcome of the cancel, either VNCDiscoverySDKErrorNone or some
 *         other error code.
 *
 * \see VNCDiscoverySDKError
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscovererCancelRequest(VNCDiscoverer *pDiscoverer, 
                            VNCDiscoverySDKRequestId requestId);

/**
 * \brief Destroys an entity of the Discoverer instance.
 *
 * The Discoverer owns all the entities it discovers, but it should not destroy
 * them (and free the memory) before the SDK asks them to. This method is used
 * to ask the instance to destroy the specific entity.
 *
 * \param pDiscoverer The Discoverer instance that owns the entity.
 * \param pEntity The entity that needs to be destroyed.
 */
typedef void VNCCALL
VNCDiscovererDestroyEntity(VNCDiscoverer *pDiscoverer,
                VNCDiscovererEntity *pEntity);

/**
 * \brief Creates in memory an entity.
 *
 * This is a method that needs to be implemented by an Address-book-like
 * Discoverer. It should create in memory a new Entity. The Application
 * can use this entity to set various key-value pairs and then the entity can
 * be published.
 *
 * Most Discoverers would choose not to implement this method. The client of
 * the SDK will be informed by the SDK that this feature is not supported.
 * 
 * An entity that has been created is owned by the Discoverer and it should
 * live in memory until either the Entity, or the Discoverer is destroyed.
 *
 * \param pDiscoverer The Discoverer instance which will own the entity.
 * \param pEntity This is where the pointer to the newly created entity will be
 *                passed back. If the entity can't be created the pointer
 *                should be NULL.
 *
 * \return VNCDiscoverySDKErrorNone if the entity was created successfully,
 *         VNCDiscoverySDKErrorOutOfMemory if there is not enough memory to
 *         create the entity, or another Discoverer-specific error code.
 *
 * \see VNCDiscoverySDKError
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscovererCreateEntity(VNCDiscoverer *pDiscoverer,
                            VNCDiscovererEntity **pEntity);

/**
 * \brief Publishes an entity that has previously been created by the
 * Discoverer.
 *
 * This call is needed for an Address-book-like Discoverer and should be called
 * after the Application has created an Entity and has set all the necessary
 * key-value pairs for this entity.
 *
 * Most Discoverers would choose not to implement this method.
 *
 * This call is asynchronous and a confirmation/error will be sent through 
 * VNCDiscovererNotifyPublishedEntity from the SupportAPI.
 * 
 * \param pDiscoverer The Discoverer instance that owns the entity.
 * \param pEntity The entity that needs to be published.
 *
 * \see VNCDiscovererSupportAPI
 */
typedef void VNCCALL
VNCDiscovererPublishEntity(VNCDiscoverer *pDiscoverer,
                            VNCDiscovererEntity *pEntity);
 
/**
 * \brief Removes an entity that has previously been created by the Discoverer.
 *
 * This call is needed for an Address-book-like Discoverer and should be called
 * for an Entity that has been published/discovered.
 *
 * Most Discoverers would choose not to implement this method.
 *
 * This call is asynchronous and a confirmation/error will be sent through 
 * VNCDiscovererNotifyUnpublishedEntity from the SupportAPI.
 *
 * When an entity is unpublished it should still be kept in the memory, until
 * either the Discoverer is destroyed, or Destroy Unpublished Entity gets
 * called.
 * 
 * \param pDiscoverer The Discoverer instance that owns the entity.
 * \param pEntity The entity that needs to be unpublished (removed).
 *
 * \see VNCDiscovererSupportAPI
 */
typedef void VNCCALL
VNCDiscovererUnpublishEntity(VNCDiscoverer *pDiscoverer,
                          VNCDiscovererEntity *pEntity);

/**
 * \brief Destroys from the memory an entity that has been unpublished.
 *
 * This call is needed for an Address-book-like Discoverer. This should be
 * called for an Entity that is not published (either it never got published,
 * or unpublish has successfully been called). It is down to the Discoverer to
 * check if these conditions are met (the SDK might make this call for an
 * entity that is published at the moment).
 *
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The entity that needs to be destroyed.
 *
 * \return VNCDiscoverySDKErrorNone if the entity has been successfully
 *         destroyed, or another error code if there has been some problem.
 * \return VNCDiscoverySDKErrorNone if the entity was destroyed successfully,
 *         or another Discoverer-specific error code.
 *
 * \see VNCDiscoverySDKError
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscovererDestroyUnpublishedEntity(VNCDiscoverer *pDiscoverer,
                            VNCDiscovererEntity *pEntity);

/**
 * \brief Destroys a Discoverer instance.
 *
 * This call destroys a specific Discoverer instance. All entities owned by the 
 * Discoverer become invalid and should be destroyed themselves.
 *
 * \param pDiscoverer The Discoverer that needs to be destroyed.
 */
typedef void VNCCALL
VNCDiscovererDestroy(VNCDiscoverer *pDiscoverer);

/**
 * \brief Called by the SDK to get multiple event handles of the Discoverer.
 *
 * It is recommended to implement VNCDiscovererGetMultipleEventHandlesEx
 * instead.
 *
 * \param pDiscoverer The Discoverer for which to monitor the handles.
 *
 * \return Array of event handles of the Discoverer, the last element in the
 *    array must be VNCDiscovererNoEventHandle, and no elements before the last
 *    one should have that value (in other words it's a VNCDiscovererNoEventHandle-terminated
 *    array). The array is owned by the Discoverer, so it is the plug-in's
 *    responsibility to free any allocated memory. The Discoverer should not
 *    change the array outside calls to VNCDiscovererGetMultipleEventHandles,
 *    unless the Discoverer is stopped.
 *
 * \see VNCDiscovererGetMultipleEventHandlesEx
 * \see VNCDiscovererHandleSingleEvent, VNCDiscovererSetTimerForHandle,
 * VNCDiscovererHandleTimeoutForEventHandle
 * \see VNCDiscovererEventHandlesChanged
 * \see VNCDiscovererGetEventHandle
 */
typedef VNCDiscovererEventHandle *VNCCALL
VNCDiscovererGetMultipleEventHandles(VNCDiscoverer *pDiscoverer);

/**
 * \brief Method called by the SDK whenever and event happens on a handle
 * provided by the Discoverer.
 *
 * This method will be called to notify the Discoverer that an event happened
 * on one of the handles provided. If the Discoverer is not providing multiple
 * handles to the SDK then it doesn't need to implement this method. Otherwise an
 * implementation is necessary.
 *
 * During this call it is safe to change the event handle list, change the
 * value of this handle, or to reset the timer for the handle the Discoverer.
 *
 * This method will not be called if not using multiple handles. The Discoverer
 * must implement VNCDiscovererHandleEvent instead.
 *
 * \note If you don't stop the timer, or reset it, you will be notified when
 *      the original timeout elapses.
 *
 * \param pDiscoverer The Discoverer for which the event happened.
 * \param pHandle The handle for which the event occurred. Only the value, or
 * the timeout for this handle can be changed from this call, other handles
 * must not change from this callback.
 *
 * \see VNCDiscovererInitializeType
 * \see VNCDiscovererGetMultipleEventHandlesEx
 * \see VNCDiscovererGetEventHandle, VNCDiscovererHandleEvent
 */
typedef void VNCCALL
VNCDiscovererHandleSingleEvent(VNCDiscoverer *pDiscoverer,
    VNCDiscovererEventHandle* pHandle);

/**
 * \brief Method called by the SDK whenever the timeout for the handle
 * requested by the Discoverer is reached. 
 *
 * This method will be called to notify the Discoverer that the requested
 * timeout associated with a handle was reached. 
 *
 * If the Discoverer is not asking for timer access from the
 * SDK then it doesn't need to implement this method. Otherwise, if using
 * multiple handles, an implementation is necessary.
 *
 * During this call it is safe to change the event handle for which the timeout
 * was reached.
 *
 * If the Discoverer wishes to have further timer callbacks it must set the
 * timeout request again.
 *
 * This method will be called if using multiple handles. If not using multiple
 * handles, the Discoverer must implement
 * VNCDiscovererHandleTimeout instead.
 *
 * \param pDiscoverer The Discoverer for which the target time was reached.
 * \param pHandle The handle for which the timer expired. Only the value, or
 * the timeout for this handle can be changed from this call, other handles
 * must not change from this callback.
 *
 * \see VNCDiscovererSetTimer
 * \see VNCDiscovererGetMultipleEventHandlesEx
 * \see VNCDiscovererGetEventHandle, VNCDiscovererHandleTimeout
 */
typedef void VNCCALL
VNCDiscovererHandleTimeoutForEventHandle(VNCDiscoverer *pDiscoverer,
    VNCDiscovererEventHandle* pHandle);

/**
 * \brief Offers the Discoverer the chance to initialize any dependencies it
 * might have.
 *
 * This is called after the Discoverer has been created successfully by the
 * SDK. It gives the Discoverer the chance to make any initializations that it
 * might need, but was not able to make when being created. For example
 * VNCDiscovererSubscribeToDeviceProvider would be typically called from within
 * here, as well as any licensing checks.
 *
 * \param pDiscoverer The Discoverer that can run the initialization.
 *
 * \return An error code describing what failed during initialization.
 *
 * \see VNCDiscovererInterface
 * \see VNCDiscovererSubscribeToDeviceProvider
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscovererInit(VNCDiscoverer* pDiscoverer);

/**
 * \brief Notifies the Discoverer that a Device Provider is started.
 *
 * If a Discoverer is subscribed to a Device Provider, it notifies it that the
 * Provider is started. This will be called only after the Discoverer is
 * requested to start and the Device Provider has started. If the Provider was
 * already started at that point, the call will be issued anyway in order to
 * ensure that the Discoverer knows it has the Provider available.
 *
 * \param pDiscoverer The Discoverer that receives the notification.
 * \param pDeviceProvider The Device Provider.
 * \param error VNCDiscoverySDKErrorNone if the Provider is started
 * successfully, otherwise, if an error was encountered, the error code is
 * passed to the Discoverer.
 *
 * \see VNCDiscovererInterface
 * \see VNCDiscovererSubscribeToDeviceProvider,
 * VNCDiscovererDeviceProviderStopped
 */
typedef void VNCCALL
VNCDiscovererDeviceProviderStarted(VNCDiscoverer* pDiscoverer,
    VNCDiscovererDeviceProvider* pDeviceProvider,
    VNCDiscoverySDKError error);

/**
 * \brief Checks with the Discoverer the level to which it needs a device.
 *
 * This is called when the Discovery SDK has a device and it needs to decide to
 * what Discoverer to offer it to. The Discoverer should check the device
 * description and report how interested it is in the device. The SDK will then
 * decide which Discoverer will receive the device.
 *
 * \param pDiscoverer The Discoverer that should check if it needs the device.
 * \param pDeviceProvider The Device Provider.
 * \param pDevice The device information to be checked.
 * \param ppInterestDescription Some text explaining why the Discoverer needs
 * access to the device and what it might expect to discover on the device.
 * This text might be presented to the user if several Discoverers want access
 * to the same device.
 *
 * \return The Discoverer should return the level of interest it has in the
 * device.
 *
 * \see VNCDiscovererInterface
 * \see VNCDiscovererSubscribeToDeviceProvider, VNCDeviceInterestLevel,
 * VNCDiscovererOfferDevice
 */
typedef VNCDeviceInterestLevel VNCCALL
VNCDiscovererGetDeviceInterest(VNCDiscoverer *pDiscoverer,
    VNCDiscovererDeviceProvider* pDeviceProvider,
    const VNCDeviceInfo* pDevice,
    char **ppInterestDescription);

/**
 * \brief Offers a device to the Discoverer.
 *
 * If the Discoverer presented some interest in the device and the SDK decided
 * it can offer it to the Discoverer, then it does so. The Discoverer can use
 * the device to retrieve further information, configure it, etc.
 *
 * The Discoverer will be notified when the device is no longer present.
 *
 * The Discoverer can return the device back to the SDK by calling
 * VNCDiscovererReleaseDevice.
 *
 * \param pDiscoverer The Discoverer receiving the device.
 * \param pDeviceProvider The Provider which found the device.
 * \param pDevice The device that the Discoverer can now use.
 *
 * \see VNCDiscovererInterface
 * \see VNCDiscovererReleaseDevice, VNCDiscovererGetDeviceInterest,
 * VNCDiscovererRemoveDevice
 */
typedef void VNCCALL
VNCDiscovererOfferDevice(VNCDiscoverer *pDiscoverer,
    VNCDiscovererDeviceProvider* pDeviceProvider,
    const VNCDeviceInfo* pDevice);

/**
 * \brief Offers a device to the Discoverer specifying the protocol to be used.
 *
 * If the Discoverer presented some interest in the device and the SDK decided
 * it can offer it to the Discoverer, then it does so. The Discoverer can use
 * the device to retrieve further information, configure it, etc.
 *
 * The Discoverer will be notified when the device is no longer present.
 *
 * The Discoverer can return the device back to the SDK by calling
 * VNCDiscovererReleaseDevice, and it should do so if it does not understand
 * the VNCDiscovererOfferParameter.
 *
 * Ownership of pProtocolParam is transferred.
 *
 * \param pDiscoverer The Discoverer receiving the device.
 * \param pDeviceProvider The Provider which found the device.
 * \param pDevice The device that the Discoverer can now use.
 * \param pProtocolParam The protocol to be used as well as any extra necessary
 * parameters for the protocol.
 *
 * \see VNCDiscovererInterface
 * \see VNCDiscovererReleaseDevice, VNCDiscovererGetDeviceInterest,
 * VNCDiscovererRemoveDevice
 */
typedef void VNCCALL
VNCDiscovererOfferDeviceForProtocol(VNCDiscoverer *pDiscoverer,
    VNCDiscovererDeviceProvider* pDeviceProvider,
    const VNCDeviceInfo* pDevice,
    const VNCDiscovererOfferParameter *pProtocolParam);

/**
 * \brief Called when the device is no longer available to the Discoverer.
 *
 * If a device has been offered to the Discoverer, when no longer available,
 * this method will be called. The Discoverer might need to clear some entities
 * at this point and clean up anything related to the device.
 *
 * \param pDiscoverer The Discoverer the receives the device.
 * \param pDeviceProvider The Provider which found the device.
 * \param pDevice The device that the Discoverer can no longer use.
 *
 * \see VNCDiscovererInterface
 * \see VNCDiscovererOfferDevice, VNCDiscovererGetDeviceInterest,
 * VNCDiscovererRemoveDevice
 */
typedef void VNCCALL
VNCDiscovererRemoveDevice(VNCDiscoverer *pDiscoverer,
    VNCDiscovererDeviceProvider* pDeviceProvider,
    const VNCDeviceInfo* pDevice);

/**
 * \brief Notifies the Discoverer that a Device Provider has stopped.
 *
 * If a Discoverer is subscribed to a Device Provider, it notifies it that the
 * Provider has stopped if previously it notified it of a start.
 *
 * Once the Discoverer has stopped, this call will be issued because the Device
 * Provider is no longer considering this Discoverer.
 *
 * This call will not be issued after the Discoverer has stopped.
 *
 * \param pDiscoverer The Discoverer that receives the notification.
 * \param pProvider The Device Provider.
 * \param error VNCDiscoverySDKErrorNone if the Provider is stopped
 * successfully, otherwise, if an error occurred, the error code is passed to
 * the Discoverer.
 *
 * \see VNCDiscovererInterface
 * \see VNCDiscovererSubscribeToDeviceProvider,
 * VNCDiscovererDeviceProviderStarted
 */
typedef void VNCCALL
VNCDiscovererDeviceProviderStopped(VNCDiscoverer* pDiscoverer,
    VNCDiscovererDeviceProvider* pDeviceProvider,
    VNCDiscoverySDKError error);

/**
 * \brief Get a description of what the Discoverer does.
 *
 * The Discoverer should give details of what sort of servers it is able to
 * discover, or any other information that may be useful to the user.
 */
typedef const char *VNCCALL
VNCDiscovererGetDescription(VNCDiscoverer* pDiscoverer);

/**
 * \brief Called by the SDK to get multiple event handles of the Discoverer.
 *
 * All the handles will be monitored by the SDK and whenever an event happens
 * VNCDiscovererHandleSingleEvent will be called. This method will be called each
 * time the Discovery SDK will listen for new event. The event handle list and
 * the handle values should not change between calls to this function. If
 * a value needs to change, or the event handles list needs to be modified,
 * then VNCDiscovererEventHandlesChanged should be called by the Discoverer.
 * This will trigger VNCDiscovererGetMultipleEventHandlesEx to be called again.
 *
 * The event handles will not be monitored when the Discoverer is not started. 
 *
 * If a Discoverer does not need to pass multiple handles, then this method
 * must not be implemented, and VNCDiscovererGetEventHandle should be used
 * instead. Having both this method and VNCDiscovererGetEventHandle implemented
 * will result in the discoverer not being loaded.
 *
 * If this method is implemented, then VNCDiscovererGetMultipleEventHandles
 * must not be implemented.
 *
 * \param pDiscoverer The Discoverer for which to monitor the handles.
 * \param ppWriteNotification On return should point to an array
 * which matches the returned array of handles. Each element in the array will
 * instruct the SDK whether to listen for "write" notifications on the event
 * handle. By default all the handles are treated as "read" handles, but if the
 * corresponding value is set, the handles will be checked for "write" as well.
 * It is possible for the Discoverer not to set the array at all (which means
 * the pointer value will remain NULL). In that case it is assumed that all the
 * handles are to be checked only for read.
 *
 * \return Array of event handles of the Discoverer, the last element in the
 *    array must be VNCDiscovererNoEventHandle, and no elements before the last
 *    one should have that value (in other words it's a VNCDiscovererNoEventHandle-terminated
 *    array). The array is owned by the Discoverer, so it is the plug-in's
 *    responsibility to free any allocated memory. The Discoverer should not
 *    change the array outside calls to VNCDiscovererGetMultipleEventHandlesEx,
 *    unless the Discoverer is stopped.
 *
 * \see VNCDiscovererHandleSingleEvent, VNCDiscovererSetTimerForHandle,
 * VNCDiscovererHandleTimeoutForEventHandle
 * \see VNCDiscovererEventHandlesChanged
 * \see VNCDiscovererGetEventHandle
 */
typedef VNCDiscovererEventHandle *VNCCALL
VNCDiscovererGetMultipleEventHandlesEx(VNCDiscoverer *pDiscoverer,
    int **ppWriteNotification);

/**
 * \brief Called by the SDK when VNCDiscoverySDKDiscovererSetContext is
 * invoked.
 *
 * \param pDiscoverer The Discoverer that will use the new context
 * \param pContext The context pointer (pDiscovererContext) passed to
 * VNCDiscoverySDKDiscovererSetContext. This pointer is not modified in any way
 * by the SDK.
 *
 * \see VNCDiscovererSupportAPI
 */
typedef void VNCCALL
VNCDiscovererSetExternalContext(VNCDiscoverer *pDiscoverer, void *pContext);

/**
 * \brief The Interface that the Discoverer should implement. 
 *
 * A Discoverer doesn't need to implement all the members of the structure, 
 * but the minimum requirement is: VNCDiscovererSetContext, VNCDiscovererStart,
 * VNCDiscovererStop, VNCDiscovererFetchEntityValue and VNCDiscovererDestroy.
 */
typedef struct
{
  /* Discoverer settings */
  /**
   * Called by the SDK to set a context that will be presented in the
   * supportAPI.
   */
  VNCDiscovererSetContext *setContext;
  /**
   * Called by the SDK to get the event handle of the Discoverer.
   */
  VNCDiscovererGetEventHandle *getEventHandle;
  /**
   * Method called by the SDK whenever and event happens on the handle
   * provided by the Discoverer.
   */
  VNCDiscovererHandleEvent *handleEvent;
  /**
   * Method called by the SDK whenever the target time of the event handle
   * provided by the Discoverer is reached.
   */
  VNCDiscovererHandleTimeout *handleTimeout;
  /**
   * Called by the SDK to get the event handle of an entity.
   */
  VNCDiscovererGetEntityEventHandle *getEntityEventHandle;
  /**
   * Method called by the SDK whenever and event happens on a handle
   * provided by one of the Entities of the Discoverer.
   */
  VNCDiscovererHandleEntityEvent *handleEntityEvent;
  /**
   * Method called by the SDK whenever the target time of the event handle
   * provided by an Entity of the Discoverer is reached.
   */
  VNCDiscovererHandleEntityTimeout *handleEntityTimeout;
  /**
   * Sets a specific property on a Discoverer instance (synchronous).
   */
  VNCDiscovererSetProperty *setProperty;
  /**
   * Gets a specific property from a Discoverer instance (synchronous).
   */
  VNCDiscovererGetProperty *getProperty;
  /**
   * Retrieves synchronously an array of extra Viewer parameters used for 
   * a particular Discoverer.
   */
  VNCDiscovererGetViewerParameters *getViewerParameters;
  /* Starting/stopping */
  /**
   * Starts the discovery process.
   */
  VNCDiscovererStart *start;
  /**
   * Stops the discovery process.
   */
  VNCDiscovererStop *stop;
  /* Operations on the entity */
  /**
   * Fetches asynchronously a specific entity value.
   */
  VNCDiscovererFetchEntityValue *fetchEntityValue;
  /**
   * Gets some binary data owned by an entity.
   */
  VNCDiscovererGetEntityBinaryData *getEntityBinaryData;
  /**
   * Post asynchronously a specific entity value.
   */
  VNCDiscovererPostEntityValue *postEntityValue;
  /**
   * Cancels and outstanding fetch/post request.
   */
  VNCDiscovererCancelRequest *cancelRequest;
  /**
   * Destroys an entity of the Discoverer instance.
   */
  VNCDiscovererDestroyEntity *destroyEntity;
  /* APIs needed for the Address-book */
  /**
   * Creates in memory an entity.
   */
  VNCDiscovererCreateEntity *createEntity;
  /**
   * Publishes an entity that has previously been created by the
   * Discoverer.
   */
  VNCDiscovererPublishEntity *publishEntity;
  /**
   * Removes an entity that has previously been created by the Discoverer.
   */
  VNCDiscovererUnpublishEntity *unpublishEntity;
  /**
   * Destroys from the memory an entity that has been unpublished.
   */
  VNCDiscovererDestroyUnpublishedEntity *destroyUnpublishedEntity;
  /* Destroying the Discoverer */
  /**
   * Destroys a Discoverer instance.
   */
  VNCDiscovererDestroy *destroy;
  /**
   * Use VNCDiscovererGetMultipleEventHandlesEx instead.
   */
  VNCDiscovererGetMultipleEventHandles *getMultipleEventHandles;
  /**
   * Method called by the SDK whenever and event happens on a handle provided
   * by the Discoverer.
   */
  VNCDiscovererHandleSingleEvent *handleSingleEvent;
  /**
   * Method called by the SDK whenever the timeout for the handle requested by
   * the Discoverer is reached. 
   */
  VNCDiscovererHandleTimeoutForEventHandle *handleTimeoutForEventHandle;
  /**
   * Offers the Discoverer the chance to initialize any dependencies it
   * might have.
   */
  VNCDiscovererInit *init;
  /**
   * Notifies the Discoverer that a Device Provider is started.
   */
  VNCDiscovererDeviceProviderStarted *deviceProviderStarted;
  /**
   * Checks with the Discoverer the level to which it needs a device.
   */
  VNCDiscovererGetDeviceInterest *getDeviceInterest;
  /**
   * Offers a device to the Discoverer.
   */
  VNCDiscovererOfferDevice *offerDevice;
  /**
   * Called when the device is no longer available to the Discoverer.
   */
  VNCDiscovererRemoveDevice *removeDevice;
  /**
   * Notifies the Discoverer that a Device Provider has stopped.
   */
  VNCDiscovererDeviceProviderStopped *deviceProviderStopped;
  /**
   * Get a description of what the Discoverer does.
   */
  VNCDiscovererGetDescription *getDescription;
  /**
   * Called by the SDK to get multiple event handles of the Discoverer.
   */
  VNCDiscovererGetMultipleEventHandlesEx *getMultipleEventHandlesEx;
  /**
   * Called by the SDK when VNCDiscoverySDKDiscovererSetContext is invoked.
   */
  VNCDiscovererSetExternalContext *setExternalContext;
  /**
   * Offers a device to the Discoverer specifying the protocol.
   */
  VNCDiscovererOfferDeviceForProtocol *offerDeviceForProtocol;
} VNCDiscovererInterface;

/**
 * \brief The type of the sole entry point to be exported by the Discoverer DLL
 * or shared object.
 *
 * \param pInterface A pointer to the structure that needs to be populated with
 *                   the interface.
 * \param interfaceSize The size of the interface structure.
 * \param pSupportAPI A pointer to the Support API structure provided for the
 *                    Discoverer.
 * \param supportAPISize The size of the support API structure.
 *
 * \return A new instance of the Discoverer, or NULL if an error occurred.
 *
 * \see VNCDiscovererInterface, VNCDiscovererSupportAPI
 */
typedef VNCDiscoverer *VNCCALL
VNCDiscovererInitializeType(VNCDiscovererInterface *pInterface,
            size_t interfaceSize,
            VNCDiscovererSupportAPI *pSupportAPI,
            size_t supportAPISize);

#ifdef __cplusplus
}
#endif


#endif /* !defined(__VNCDISCOVERER_H__) */

